;
; Basic Windows(R) 32bit Structured Exception Handling and Hardware Breakpoint example in FASM
; (C) 2006 Bryan "RedGhost" Power
;
; Explaination:
;     When an exception occurs (invalid memory reference, software breakpoint, etc), Windows(R) handles this in ring0 before
;     eventually dispatching to KiUserExceptionDispatcher located in ntdll.dll, KiUserExceptionDispatcher then calls the top level
;     exception handler in the SEH chain, which is a linked list of procedure addresses.
;     The SEH chain is located in the Thread Environment Block (TEB).
;     
;     Inside an exception handler you can identify exactly what went wrong and possibly even continue programme execution.
;
;     For more information please check out more detailed articles by Matt Pietrek @ the Microsoft(R) developers network.
;

format PE GUI

include 'seh_data.inc'
include 'win32a.inc'

section '.flat' readable writeable executable

entry $ 					; entry point at current address
    push  exception_handler
    call  [SetUnhandledExceptionFilter] 	; add our handler to the top of the SEH chain, function doesn't fail

    ; software breakpoint generating an EXCEPTION_BREAKPOINT
    int3

    ; right here we return from the handler after the first exception

    ; try and access invalid memory generating an EXCEPTION_ACCESS_VIOLATION
    xor   eax, eax
    mov   dword [eax], $FFFFFFFF

    push  MB_OK
    push  _title
    push  _crash
    push  $0
    call  [MessageBoxA]

    ; now lets set a hardware breakpoint
    mov   dword [ctx.ContextFlags], CONTEXT_DEBUG_REGISTERS
    mov   dword [ctx.Dr0], @f			; address of first breakpoint
    mov   dword [ctx.Dr7], $00000001		; execute breakpoint (not r/w) on the address in Dr0

    call  [GetCurrentThread]			; pseudo thread handle
    push  ctx
    push  eax
    call  [SetThreadContext]

    ; mess with the registers to switch up the displayed info
    mov   eax, $1337
    mov   ebx, $80386
    mov   ecx, $FA5AA
    mov   edx, $C0D3
    xor   edi, edi

    ; when the instruction pointer reaches this address an exception will be thrown
@@:
    ; this shouldn't be called
    push  $0
    call  [ExitProcess]
;---

; this should be modeled after "UnhandledExceptionFilter"
; LONG WINAPI UnhandledExceptionFilter( struct _EXCEPTION_POINTERS* ExceptionInfo );
exception_handler:
    push  ebp
    mov   ebp, esp

    push  ebx edi

    mov   eax, [ebp+$8] 			; get the EXCEPTION_POINTERS structure
    mov   ebx, [eax+EXCEPTION_POINTERS.ExceptionRecord]
    mov   edi, [eax+EXCEPTION_POINTERS.ContextRecord]

    ; format the display string
    push  dword [edi+CONTEXT.Eip]
    push  dword [edi+CONTEXT.Esp]
    push  dword [edi+CONTEXT.Ebp]
    push  dword [edi+CONTEXT.Esi]
    push  dword [edi+CONTEXT.Edi]
    push  dword [edi+CONTEXT.Edx]
    push  dword [edi+CONTEXT.Ecx]
    push  dword [edi+CONTEXT.Ebx]
    push  dword [edi+CONTEXT.Eax]
    push  dword [ebx+EXCEPTION_RECORD.ExceptionAddress]
    push  dword [ebx+EXCEPTION_RECORD.ExceptionCode]
    push  _format
    push  _buf
    call  [wsprintfA]
    add   esp, $34				; balance the stack after a cdecl call

    ; display the info
    push  MB_OK
    push  _title
    push  _buf
    push  $0
    call  [MessageBoxA]

    ; check for a software breakpoint
    cmp   dword [ebx+EXCEPTION_RECORD.ExceptionCode], EXCEPTION_BREAKPOINT
    jz	  .breakpoint

    cmp   dword [ebx+EXCEPTION_RECORD.ExceptionCode], EXCEPTION_ACCESS_VIOLATION
    jz	  .accessviolation

    ; here we have our EXCEPTION_SINGLE_STEP from the hardware breakpoint
    mov   eax, EXCEPTION_CONTINUE_SEARCH	; continue the search until the OS catches it
    jmp   .done

.breakpoint:
    ; it is our int3 breakpoint
    inc   dword [edi+CONTEXT.Eip]		; increment EIP to skip the int3 instruction and avoid breakpoint recursion when we continue execution
    mov   eax, EXCEPTION_CONTINUE_EXECUTION
    jmp   .done

.accessviolation:
    ; EXCEPTION_ACCESS_VIOLATION
    mov   dword [edi+CONTEXT.Eax], valid_memory ; change eax so it points to valid memory and continue execution
    mov   eax, EXCEPTION_CONTINUE_EXECUTION

.done:
    pop   edi  ebx

    mov   esp, ebp
    pop   ebp
    ret   $4
;---

_title	db 'SEH example in FASM', $0
_crash	db 'We are going to set a hardware breakpoint, the programme will crash after the next MessageBoxA call.', $0
_format db 'Exception Info', $D, $A, $D, $A,\
	   'Exception Code:', $9, '0x%X', $D, $A,\
	   'Exception Address:', $9, '0x%X', $D, $A, $D, $A,\
	   'Register Values On Exception', $D, $A, $D, $A,\
	   'eax:', $9, '0x%X', $D, $A,\
	   'ebx:', $9, '0x%X', $D, $A,\
	   'ecx:', $9, '0x%X', $D, $A,\
	   'edx:', $9, '0x%X', $D, $A,\
	   'edi:', $9, '0x%X', $D, $A,\
	   'esi:', $9, '0x%X', $D, $A,\
	   'ebp:', $9, '0x%X', $D, $A,\
	   'esp:', $9, '0x%X', $D, $A,\
	   'eip:', $9, '0x%X', $0
_buf	rb $100

align 4
valid_memory dd ?
ctx	     CONTEXT

data import
    dd $0, $0, $0, rva kernel32_name, rva kernel32_table
    dd $0, $0, $0, rva user32_name,   rva user32_table
    dd $0, $0, $0, $0, $0

kernel32_table:
    SetUnhandledExceptionFilter  dd rva _SetUnhandledExceptionFilter
    GetCurrentThread		 dd rva _GetCurrentThread
    SetThreadContext		 dd rva _SetThreadContext
    ExitProcess 		 dd rva _ExitProcess
    dd $0

user32_table:
    wsprintfA			 dd rva _wsprintfA
    MessageBoxA 		 dd rva _MessageBoxA
    dd $0

; kernel32
    _SetUnhandledExceptionFilter dw $0
				 db 'SetUnhandledExceptionFilter', $0
    _GetCurrentThread		 dw $0
				 db 'GetCurrentThread', $0
    _SetThreadContext		 dw $0
				 db 'SetThreadContext', $0
    _ExitProcess		 dw $0
				 db 'ExitProcess', $0

; user32
    _wsprintfA			 dw $0
				 db 'wsprintfA',   $0
    _MessageBoxA		 dw $0
				 db 'MessageBoxA', $0

    kernel32_name db 'kernel32.dll', $0
    user32_name   db 'user32.dll', $0
end data
;---

; bye




